<?php

namespace Omnipay\Aifo\Message;

use Omnipay\Common\Exception\InvalidResponseException;
use Omnipay\Common\Message\AbstractResponse;
use Omnipay\Common\Message\RequestInterface;

class CompletePurchaseResponse extends AbstractResponse
{
    protected $request;

    public function __construct(RequestInterface $request, $data)
    {
        $this->request = $request;
        $this->data = $data;

        if (empty($this->data['sum']) || empty($this->data['invoice']) || empty($this->data['http_auth_signature'])) {
            throw new InvalidResponseException('Missing required callback parameters: sum, invoice, http_auth_signature');
        }

        if ($this->getHttpAuthSignature() !== $this->calculateSignature()) {
            throw new InvalidResponseException('Invalid signature in callback');
        }
    }

    public function calculateSignature(): string
    {
        $shopId = (string)$this->request->getShopId();
        $amount = (float)($this->data['sum'] ?? 0);
        $secretKey = (string)$this->request->getSecretKey();
        $invoiceUid = (string)($this->data['invoice'] ?? '');
        $signatureType = (int)$this->request->getSignatureType();

        $signString = "{$shopId}:{$amount}:{$secretKey}:{$invoiceUid}";
        
        switch ($signatureType) {
            case 1: 
                return hash('md5', $signString);
            case 2: 
                return hash('sha256', $signString);
            case 3: 
                return hash('sha1', $signString);
            case 4: 
                return hash('ripemd160', $signString);
            case 5: 
                return hash('sha384', $signString);
            case 6: 
                return hash('sha512', $signString);
            default:
                return hash('sha256', $signString); 
        }
    }

    public function getHttpAuthSignature(): string
    {
        return (string)($this->data['http_auth_signature'] ?? '');
    }

    public function isSuccessful(): bool
    {
        return true;
    }

    public function getTransactionId(): ?string
    {
        $invoiceUid = $this->data['invoice'] ?? null;
        return $invoiceUid !== null ? (string)$invoiceUid : null;
    }

    public function getTransactionReference(): ?string
    {
        $invoiceUid = $this->data['invoice'] ?? null;
        return $invoiceUid !== null ? (string)$invoiceUid : null;
    }

    public function getAmount(): ?float
    {
        return (float)($this->data['sum'] ?? 0);
    }
}
