<?php
/**
 * AIFO Payment Form Generator
 * Генератор платіжної форми для AIFO.PRO
 * 
 * Використання:
 * 1. Вкажіть ваші дані в конфігурації нижче
 * 2. Відкрийте файл в браузері або підключіть до вашого сайту
 * 
 * @version 1.0
 * @author AIFO.PRO
 */

// ============================================
// КОНФІГУРАЦІЯ
// ============================================

// ID вашої каси (shop_id)
// Можна отримати в налаштуваннях проєкту на https://aifo.pro/kassa
define('SHOP_ID', '1'); // Замініть на ваш ID каси

// Секретний ключ каси (kassa_secretkey)
// НЕ публікуйте цей ключ! Він доступний тільки вам в налаштуваннях каси
define('SECRET_KEY', 'your_secret_key_here'); // Замініть на ваш секретний ключ

// URL платіжної системи
define('PAYMENT_URL', 'https://aifo.pro/pay/');

// Тип підпису (алгоритм хешування)
// 1 = MD5 (застарілий, не рекомендується)
// 2 = SHA256 (РЕКОМЕНДОВАНО)
// 3 = SHA1
// 4 = RIPEMD160
// 5 = SHA384
// 6 = SHA512
define('SIGNATURE_TYPE', 2); // SHA256 (рекомендовано)

// URL для перенаправлення після успішної оплати
define('SUCCESS_URL', 'https://your-site.com/success.php');

// URL для перенаправлення після помилки оплати
define('FAIL_URL', 'https://your-site.com/fail.php');

// ============================================
// ФУНКЦІЇ
// ============================================

/**
 * Генерація підпису для платежу
 * 
 * @param int $shop_id ID каси
 * @param float $amount Сума платежу
 * @param string $secret_key Секретний ключ
 * @param string $pay_id Номер рахунку
 * @param int $signature_type Тип підпису (1-6)
 * @return string Підпис
 */
function generateSignature($shop_id, $amount, $secret_key, $pay_id, $signature_type = 2) {
	// Формуємо рядок для підпису: "ID Магазину:Сума:секретний ключ:Номер Рахунку"
	$sign_string = $shop_id . ':' . $amount . ':' . $secret_key . ':' . $pay_id;
	
	// Генеруємо підпис залежно від типу
	switch ($signature_type) {
		case 1: // MD5 (застарілий)
			return hash('md5', $sign_string);
		case 2: // SHA256 (РЕКОМЕНДОВАНО)
			return hash('sha256', $sign_string);
		case 3: // SHA1
			return hash('sha1', $sign_string);
		case 4: // RIPEMD160
			return hash('ripemd160', $sign_string);
		case 5: // SHA384
			return hash('sha384', $sign_string);
		case 6: // SHA512
			return hash('sha512', $sign_string);
		default:
			return hash('sha256', $sign_string); // За замовчуванням SHA256
	}
}

/**
 * Генерація URL для платежу
 * 
 * @param int $shop_id ID каси
 * @param string $pay_id Номер рахунку
 * @param float $amount Сума платежу
 * @param string $secret_key Секретний ключ
 * @param int $signature_type Тип підпису
 * @param string $desc Опис платежу (опціонально)
 * @return string URL для перенаправлення
 */
function generatePaymentUrl($shop_id, $pay_id, $amount, $secret_key, $signature_type = 2, $desc = '') {
	$sign = generateSignature($shop_id, $amount, $secret_key, $pay_id, $signature_type);
	
	$params = array(
		'shop_id' => $shop_id,
		'pay_id' => $pay_id,
		'amount' => $amount,
		'sign' => $sign
	);
	
	if (!empty($desc)) {
		$params['desc'] = urlencode($desc);
	}
	
	return PAYMENT_URL . '?' . http_build_query($params);
}

/**
 * Перевірка IP адреси сервера AIFO
 * Рекомендується використовувати для безпеки
 * 
 * @return bool true якщо IP валідний
 */
function validateAifoIp() {
	$allowed_ips = array('77.83.102.155');
	
	// Отримуємо IP адресу
	$ip = '';
	if (isset($_SERVER['HTTP_CF_CONNECTING_IP'])) {
		$ip = $_SERVER['HTTP_CF_CONNECTING_IP'];
	} elseif (isset($_SERVER['HTTP_X_REAL_IP'])) {
		$ip = $_SERVER['HTTP_X_REAL_IP'];
	} elseif (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
		$ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
		$ip = trim($ips[0]);
	} else {
		$ip = $_SERVER['REMOTE_ADDR'] ?? '';
	}
	
	return in_array($ip, $allowed_ips);
}

/**
 * Перевірка підпису від AIFO (для обробки сповіщень)
 * 
 * @param array $post POST дані від AIFO
 * @param string $secret_key Секретний ключ
 * @param int $signature_type Тип підпису
 * @return bool true якщо підпис валідний
 */
function verifySignature($post, $secret_key, $signature_type = 2) {
	if (!isset($post['sum']) || !isset($post['invoice']) || !isset($post['http_auth_signature'])) {
		return false;
	}
	
	$shop_id = SHOP_ID; // Можна отримати з БД або передати як параметр
	$amount = $post['sum'];
	$pay_id = $post['invoice'];
	
	// Генеруємо очікуваний підпис
	$expected_sign = generateSignature($shop_id, $amount, $secret_key, $pay_id, $signature_type);
	
	// Порівнюємо з отриманим підписом
	return hash_equals($expected_sign, $post['http_auth_signature']);
}

// ============================================
// ПРИКЛАД ВИКОРИСТАННЯ
// ============================================

// Якщо це POST запит для обробки сповіщення від AIFO
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['http_auth_signature'])) {
	// Перевіряємо IP (рекомендується)
	if (!validateAifoIp()) {
		die('Invalid IP address');
	}
	
	// Перевіряємо підпис
	if (!verifySignature($_POST, SECRET_KEY, SIGNATURE_TYPE)) {
		die('Invalid signature');
	}
	
	// Обробка успішного платежу
	$invoice_id = $_POST['invoice'];
	$amount = $_POST['sum'];
	$signature = $_POST['http_auth_signature'];
	
	// Тут ваша логіка обробки платежу
	// Наприклад: оновлення статусу в БД, відправка email тощо
	
	echo "OK"; // Відповідь для AIFO
	exit;
}

// Якщо це GET запит - показуємо форму
?>
<!DOCTYPE html>
<html lang="uk">
<head>
	<meta charset="UTF-8">
	<meta name="viewport" content="width=device-width, initial-scale=1.0">
	<title>Платіжна форма AIFO</title>
	<style>
		* {
			margin: 0;
			padding: 0;
			box-sizing: border-box;
		}
		
		body {
			font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
			background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
			min-height: 100vh;
			display: flex;
			align-items: center;
			justify-content: center;
			padding: 20px;
		}
		
		.payment-container {
			background: white;
			border-radius: 12px;
			box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
			padding: 40px;
			max-width: 500px;
			width: 100%;
		}
		
		.payment-header {
			text-align: center;
			margin-bottom: 30px;
		}
		
		.payment-header h1 {
			color: #333;
			font-size: 28px;
			margin-bottom: 10px;
		}
		
		.payment-header p {
			color: #666;
			font-size: 14px;
		}
		
		.form-group {
			margin-bottom: 20px;
		}
		
		.form-group label {
			display: block;
			margin-bottom: 8px;
			color: #333;
			font-weight: 500;
			font-size: 14px;
		}
		
		.form-group input,
		.form-group textarea {
			width: 100%;
			padding: 12px 15px;
			border: 2px solid #e0e0e0;
			border-radius: 8px;
			font-size: 16px;
			transition: border-color 0.3s;
		}
		
		.form-group input:focus,
		.form-group textarea:focus {
			outline: none;
			border-color: #667eea;
		}
		
		.form-group small {
			display: block;
			margin-top: 5px;
			color: #999;
			font-size: 12px;
		}
		
		.btn {
			width: 100%;
			padding: 14px;
			background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
			color: white;
			border: none;
			border-radius: 8px;
			font-size: 16px;
			font-weight: 600;
			cursor: pointer;
			transition: transform 0.2s, box-shadow 0.2s;
		}
		
		.btn:hover {
			transform: translateY(-2px);
			box-shadow: 0 5px 20px rgba(102, 126, 234, 0.4);
		}
		
		.btn:active {
			transform: translateY(0);
		}
		
		.info-box {
			background: #f0f4ff;
			border-left: 4px solid #667eea;
			padding: 15px;
			border-radius: 6px;
			margin-bottom: 20px;
		}
		
		.info-box p {
			margin: 0;
			color: #555;
			font-size: 14px;
			line-height: 1.6;
		}
		
		.info-box strong {
			color: #667eea;
		}
		
		.error {
			background: #fee;
			border-left-color: #f00;
			color: #c00;
		}
		
		.success {
			background: #efe;
			border-left-color: #0a0;
			color: #060;
		}
	</style>
</head>
<body>
	<div class="payment-container">
		<div class="payment-header">
			<h1>💳 Платіжна форма</h1>
			<p>Безпечна оплата через AIFO.PRO</p>
		</div>
		
		<?php
		// Перевірка конфігурації
		if (SECRET_KEY === 'your_secret_key_here' || empty(SECRET_KEY)) {
			echo '<div class="info-box error">';
			echo '<p><strong>Помилка конфігурації!</strong><br>';
			echo 'Будь ласка, вкажіть ваш SECRET_KEY в файлі payment_form.php</p>';
			echo '</div>';
		}
		?>
		
		<form method="GET" action="<?php echo PAYMENT_URL; ?>" id="paymentForm">
			<div class="form-group">
				<label for="shop_id">ID Каси (Shop ID):</label>
				<input type="text" id="shop_id" name="shop_id" value="<?php echo htmlspecialchars(SHOP_ID); ?>" required>
				<small>ID вашої каси з налаштувань AIFO</small>
			</div>
			
			<div class="form-group">
				<label for="pay_id">Номер рахунку (Pay ID):</label>
				<input type="text" id="pay_id" name="pay_id" value="<?php echo 'ORDER_' . time(); ?>" required>
				<small>Унікальний номер рахунку (генерується автоматично)</small>
			</div>
			
			<div class="form-group">
				<label for="amount">Сума платежу (₴):</label>
				<input type="number" id="amount" name="amount" step="0.01" min="0.01" value="100.00" required>
				<small>Мінімальна сума: 0.01 ₴</small>
			</div>
			
			<div class="form-group">
				<label for="desc">Опис платежу (опціонально):</label>
				<textarea id="desc" name="desc" rows="2" placeholder="Опис товару або послуги"></textarea>
			</div>
			
			<!-- Приховані поля для підпису -->
			<input type="hidden" id="sign" name="sign" value="">
			
			<div class="info-box">
				<p><strong>ℹ️ Інформація:</strong><br>
				Після натискання кнопки ви будете перенаправлені на сторінку оплати AIFO.PRO</p>
			</div>
			
			<button type="submit" class="btn" id="submitBtn">Перейти до оплати</button>
		</form>
		
		<div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #e0e0e0;">
			<h3 style="font-size: 16px; margin-bottom: 15px; color: #333;">Приклад коду для інтеграції:</h3>
			<pre style="background: #f5f5f5; padding: 15px; border-radius: 6px; overflow-x: auto; font-size: 12px; line-height: 1.5;"><code><?php
$example_shop_id = SHOP_ID;
$example_pay_id = 'ORDER_12345';
$example_amount = 100.00;
$example_secret = 'your_secret_key';
$example_sign = generateSignature($example_shop_id, $example_amount, $example_secret, $example_pay_id, SIGNATURE_TYPE);
$example_url = generatePaymentUrl($example_shop_id, $example_pay_id, $example_amount, $example_secret, SIGNATURE_TYPE, 'Опис платежу');

echo htmlspecialchars("<?php
// Підключення файлу
require_once 'modules/payment_form.php';

// Параметри платежу
\$shop_id = " . $example_shop_id . ";
\$pay_id = 'ORDER_12345'; // Унікальний номер рахунку
\$amount = 100.00; // Сума в гривнях
\$desc = 'Опис платежу'; // Опціонально

// Генерація URL для оплати
\$payment_url = generatePaymentUrl(
    \$shop_id, 
    \$pay_id, 
    \$amount, 
    SECRET_KEY, 
    SIGNATURE_TYPE, 
    \$desc
);

// Перенаправлення на сторінку оплати
header('Location: ' . \$payment_url);
exit;
?>");
?></code></pre>
		</div>
	</div>
	
	<script>
		// Генерація підпису на стороні клієнта (для демонстрації)
		// УВАГА: В реальному застосунку підпис має генеруватися на сервері!
		document.getElementById('paymentForm').addEventListener('submit', function(e) {
			var shopId = document.getElementById('shop_id').value;
			var payId = document.getElementById('pay_id').value;
			var amount = document.getElementById('amount').value;
			
			// Перевірка валідності
			if (!shopId || !payId || !amount || parseFloat(amount) <= 0) {
				e.preventDefault();
				alert('Будь ласка, заповніть всі обов\'язкові поля правильно!');
				return false;
			}
			
			// У реальному застосунку тут має бути AJAX запит на сервер для генерації підпису
			// Для демонстрації використовуємо PHP генерацію
			// Підпис буде додано через PHP нижче
		});
		
		// Автоматична генерація унікального Pay ID
		document.getElementById('pay_id').value = 'ORDER_' + Date.now();
	</script>
	
	<?php
	// Генерація підпису при відправці форми
	if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['shop_id']) && isset($_GET['pay_id']) && isset($_GET['amount'])) {
		$shop_id = $_GET['shop_id'];
		$pay_id = $_GET['pay_id'];
		$amount = floatval($_GET['amount']);
		$desc = isset($_GET['desc']) ? $_GET['desc'] : '';
		
		// Генеруємо підпис
		$sign = generateSignature($shop_id, $amount, SECRET_KEY, $pay_id, SIGNATURE_TYPE);
		
		// Формуємо URL
		$payment_url = generatePaymentUrl($shop_id, $pay_id, $amount, SECRET_KEY, SIGNATURE_TYPE, $desc);
		
		// Перенаправляємо на сторінку оплати
		header('Location: ' . $payment_url);
		exit;
	}
	?>
</body>
</html>
