# Telegram бот для продажу цифрових товарів
# Інтеграція з AIFO Payment System
import telebot
from telebot import types
import datetime
import requests
import json
import logging
import hashlib
import time
from config import TOKEN, AIFO_API_URL, AIFO_SHOP_ID, AIFO_SECRET_KEY, ADMIN_ID, CHANNEL_ID
from tinydb import TinyDB, Query

logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler('bot.log', encoding='utf-8'),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger(__name__)

bot = telebot.TeleBot(TOKEN)

try:
    db = TinyDB('database.json', ensure_ascii=False)
    users = db.table('users')
    categories = db.table('categories')
    products = db.table('products')
    transactions = db.table('transactions')
    orders = db.table('orders')
    logger.info("База даних успішно ініціалізована")
except Exception as e:
    logger.error(f"Помилка ініціалізації бази даних: {e}")
    raise

FIRE_EFFECT = '5104841245755180586'
PARTY_EFFECT = '5046509860389126442'


def create_signature(shop_id, amount, secret_key, invoice_id):
    """
    Створення підпису для AIFO API
    Формула: SHA256(shop_id:amount:secret_key:invoice_id)
    
    PHP при конкатенації (float)$amount використовує стандартне форматування:
    - (float)100.0 -> "100" (без .0)
    - (float)100.5 -> "100.5" (без зайвих нулів)
    """
    amount_float = float(amount)
    # Для цілих чисел використовуємо int (як PHP)
    if amount_float.is_integer():
        amount_formatted = str(int(amount_float))
    else:
        # Для дробових використовуємо стандартний формат Python (як PHP)
        amount_formatted = str(amount_float)
    sign_string = f"{shop_id}:{amount_formatted}:{secret_key}:{invoice_id}"
    return hashlib.sha256(sign_string.encode()).hexdigest()


def send_api_request(endpoint, data, method='POST'):
    """
    Безпечна відправка запиту до AIFO API з обробкою помилок
    """
    try:
        url = f"{AIFO_API_URL}/{endpoint}"
        headers = {'Content-Type': 'application/json'}
        
        logger.info(f"Відправка запиту до {url} з даними: {json.dumps(data, ensure_ascii=False)}")
        
        if method == 'POST':
            response = requests.post(url, json=data, headers=headers, timeout=10)
        else:
            response = requests.get(url, params=data, headers=headers, timeout=10)
        
        logger.info(f"Статус відповіді: {response.status_code}")
        logger.info(f"Відповідь API: {response.text[:500]}")
        
        response.raise_for_status()
        result = response.json()
        
        if result.get('status') == 'success':
            return result.get('data', {})
        else:
            error_msg = result.get('message', 'Невідома помилка API')
            error_data = result.get('error', '')
            logger.error(f"API помилка: {error_msg}, дані: {error_data}")
            return {'error': error_msg, 'error_data': error_data}
            
    except requests.exceptions.Timeout:
        logger.error(f"Таймаут запиту до {endpoint}")
        return {'error': 'Таймаут запиту до API'}
    except requests.exceptions.HTTPError as e:
        error_text = ''
        try:
            error_text = e.response.text[:1000]
            # Спробуємо парсити JSON відповідь
            try:
                error_json = e.response.json()
                error_msg_from_api = error_json.get('message', '')
                logger.error(f"HTTP помилка {e.response.status_code}: {error_msg_from_api}")
            except:
                pass
        except:
            pass
        logger.error(f"HTTP помилка {e.response.status_code} при запиті до {endpoint}")
        logger.error(f"Відповідь сервера: {error_text}")
        return {'error': f'HTTP помилка {e.response.status_code}', 'error_data': error_text}
    except requests.exceptions.RequestException as e:
        logger.error(f"Помилка запиту до {endpoint}: {e}")
        return {'error': f'Помилка підключення: {str(e)}'}
    except Exception as e:
        logger.error(f"Невідома помилка при запиті до {endpoint}: {e}", exc_info=True)
        return {'error': f'Невідома помилка: {str(e)}'}


@bot.message_handler(commands=['admin'])
def admin_panel(message):
    """Адмін панель"""
    if message.from_user.id != ADMIN_ID:
        bot.send_message(message.chat.id, "❌ У вас немає доступу до цієї команди")
        return
    
    markup = types.InlineKeyboardMarkup()
    markup.add(types.InlineKeyboardButton("📊 Статистика", callback_data="admin_stats"))
    markup.add(types.InlineKeyboardButton("📦 Товари", callback_data="admin_products_1"))
    markup.add(types.InlineKeyboardButton("➕ Додати товар", callback_data="admin_add_prod"),
              types.InlineKeyboardButton("🗑 Видалити товар", callback_data="admin_del_prod"))
    markup.add(types.InlineKeyboardButton("➕ Додати категорію", callback_data="admin_add_cat"),
              types.InlineKeyboardButton("🗑 Видалити категорію", callback_data="admin_del_cat"))
    
    bot.send_message(message.chat.id,
                    "🔧 <b>Адмін меню</b>",
                    parse_mode='HTML',
                    reply_markup=markup,
                    message_effect_id=FIRE_EFFECT)


@bot.message_handler(commands=['start'])
def send_welcome(message):
    """Привітання нового користувача"""
    user_id = message.from_user.id
    username = message.from_user.username or f"user_{user_id}"
    
    try:
        if not users.contains(Query().user_id == user_id):
            users.insert({
                'user_id': user_id,
                'username': username,
                'reg_date': datetime.datetime.now().strftime('%Y-%m-%d'),
                'balance': 0.0,
                'purchases': 0
            })
            logger.info(f"Новий користувач зареєстровано: {user_id} (@{username})")

        markup = types.ReplyKeyboardMarkup(resize_keyboard=True)
        markup.add(types.KeyboardButton("📦 Товари"), types.KeyboardButton("📁 Наявність"))
        markup.add(types.KeyboardButton("👤 Профіль"))

        bot.send_message(message.chat.id, "⚡️")
        bot.send_message(message.chat.id,
                        f"<b>🎉 Привіт, @{username}, дякую що завітав!</b>\n\n"
                        f"<b>😎 До речі, ти з нами вже з</b> <code>{datetime.datetime.now().strftime('%Y-%m-%d')}</code>",
                        parse_mode='HTML',
                        reply_markup=markup,
                        message_effect_id=FIRE_EFFECT)
    except Exception as e:
        logger.error(f"Помилка в send_welcome: {e}")
        bot.send_message(message.chat.id, "❌ Помилка при обробці команди. Спробуйте пізніше.")


@bot.message_handler(content_types=['text'])
def handle_text(message):
    """Обробка текстових повідомлень"""
    try:
        if message.text == "📦 Товари":
            show_categories(message)
        elif message.text == "📁 Наявність":
            show_stock(message)
        elif message.text == "👤 Профіль":
            show_profile(message)
    except Exception as e:
        logger.error(f"Помилка в handle_text: {e}")
        bot.send_message(message.chat.id, "❌ Помилка при обробці запиту. Спробуйте пізніше.")




def send_file_by_id(chat_id, file_id):
    """Відправка файлу за file_id (підтримує різні типи файлів)"""
    try:
        # Спроба відправити як документ (найуніверсальніший спосіб)
        bot.send_document(chat_id, file_id)
    except Exception as e:
        logger.error(f"Помилка відправки файлу: {e}")
        # Якщо не вийшло як документ, спробуємо інші типи
        try:
            bot.send_photo(chat_id, file_id)
        except:
            try:
                bot.send_video(chat_id, file_id)
            except:
                try:
                    bot.send_audio(chat_id, file_id)
                except:
                    bot.send_message(chat_id, "❌ Помилка відправки файлу")


def show_profile(message):
    """Відображення профілю користувача"""
    try:
        user_id = message.from_user.id
        user = users.get(Query().user_id == user_id)
        
        if not user:
            bot.send_message(message.chat.id, "❌ Користувача не знайдено")
            return
            
        username = user.get('username', f"user_{user_id}")
        
        markup = types.InlineKeyboardMarkup()
        markup.add(types.InlineKeyboardButton("📜 Історія замовлень", callback_data="history"))
        markup.add(types.InlineKeyboardButton("💰 Поповнити баланс", callback_data="topup"))

        bot.send_message(message.chat.id,
                        f"<b>👤 Клієнт:</b> @{username}\n"
                        f"<b>📦 Кількість покупок:</b> <code>{user.get('purchases', 0)}</code>\n"
                        f"<b>🆔 ID:</b> <code>{user_id}</code>\n"
                        f"<b>💰 Баланс:</b> <code>{user.get('balance', 0.0)} ₴</code>",
                        parse_mode='HTML',
                        reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в show_profile: {e}")
        bot.send_message(message.chat.id, "❌ Помилка при завантаженні профілю")


def show_categories(message):
    """Відображення категорій товарів"""
    try:
        cat_list = categories.all()
        
        if not cat_list:
            bot.send_message(message.chat.id, "<b>📦 Категорії відсутні</b>", parse_mode='HTML')
            return
        
        markup = types.InlineKeyboardMarkup()
        for cat in cat_list:
            markup.add(types.InlineKeyboardButton(cat['name'], callback_data=f"cat_{cat.doc_id}"))
        
        bot.send_message(message.chat.id,
                        "<b>📦 Активні категорії в магазині:</b>",
                        parse_mode='HTML',
                        reply_markup=markup,
                        message_effect_id=FIRE_EFFECT)
    except Exception as e:
        logger.error(f"Помилка в show_categories: {e}")
        bot.send_message(message.chat.id, "❌ Помилка при завантаженні категорій")


def show_categories_inline(call):
    """Відображення категорій (inline)"""
    try:
        cat_list = categories.all()
        
        markup = types.InlineKeyboardMarkup()
        for cat in cat_list:
            markup.add(types.InlineKeyboardButton(cat['name'], callback_data=f"cat_{cat.doc_id}"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="<b>📦 Активні категорії в магазині:</b>",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в show_categories_inline: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при завантаженні категорій")


def show_stock(message):
    """Відображення наявності товарів"""
    try:
        prod_list = products.all()
        if not prod_list:
            bot.send_message(message.chat.id, "<b>😔 Нічого немає :(</b>", parse_mode='HTML')
            return
        
        text = ""
        current_category = None
        for prod in prod_list:
            try:
                cat = categories.get(doc_id=prod['category_id'])
                if current_category != cat['name']:
                    text += f"<b>➖➖➖{cat['name']}➖➖➖</b>\n"
                    current_category = cat['name']
                text += f"        <b>{prod['name']}</b> | <code>{prod['price']} ₴</code>\n"
            except Exception as e:
                logger.warning(f"Помилка при обробці товару {prod.doc_id}: {e}")
                continue
        
        if text:
            bot.send_message(message.chat.id, text, parse_mode='HTML')
        else:
            bot.send_message(message.chat.id, "<b>😔 Нічого немає :(</b>", parse_mode='HTML')
    except Exception as e:
        logger.error(f"Помилка в show_stock: {e}")
        bot.send_message(message.chat.id, "❌ Помилка при завантаженні товарів")


@bot.callback_query_handler(func=lambda call: True)
def callback_handler(call):
    """Обробка callback запитів"""
    try:
        user_id = call.from_user.id
        data = call.data
        
        if data.startswith("cat_"):
            cat_id = int(data.split("_")[1])
            show_category_products(call, cat_id)
        elif data.startswith("prod_"):
            prod_id = int(data.split("_")[1])
            show_product(call, prod_id)
        elif data.startswith("buy_"):
            prod_id = int(data.split("_")[1])
            confirm_purchase(call, prod_id)
        elif data == "yes":
            complete_purchase(call)
        elif data == "back_prod":
            try:
                prod_id = int(call.message.json['reply_markup']['inline_keyboard'][0][0]['callback_data'].split('_')[1])
                prod = products.get(doc_id=prod_id)
                cat_id = prod['category_id']
                show_category_products(call, cat_id)
            except Exception as e:
                logger.error(f"Помилка в back_prod: {e}")
                bot.answer_callback_query(call.id, "❌ Помилка")
        elif data == "back_cat":
            show_categories_inline(call)
        elif data == "history":
            show_history(call)
        elif data == "topup":
            request_topup_amount(call)
        elif data == "aifo_pay":
            create_aifo_payment(call)
        elif data == "check":
            check_payment(call)
        elif data == "cancel":
            cancel_payment(call)
        elif data.startswith("admin_"):
            handle_admin(call)
        elif data.startswith("delcat_"):
            delete_category_step2(call)
        elif data.startswith("confdelcat_"):
            cat_id = int(data.split("_")[1])
            categories.remove(doc_ids=[cat_id])
            bot.edit_message_text(chat_id=call.message.chat.id, message_id=call.message.message_id, text="✅ Категорію видалено")
        elif data.startswith("addprodcat_"):
            add_product_step2(call)
        elif data.startswith("delprod_"):
            delete_product_step2(call)
        elif data.startswith("confdelprod_"):
            prod_id = int(data.split("_")[1])
            products.remove(doc_ids=[prod_id])
            bot.edit_message_text(chat_id=call.message.chat.id, message_id=call.message.message_id, text="✅ Товар видалено")
    except Exception as e:
        logger.error(f"Помилка в callback_handler: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при обробці запиту")


def show_category_products(call, cat_id):
    """Відображення товарів категорії"""
    try:
        cat = categories.get(doc_id=cat_id)
        prod_list = products.search(Query().category_id == cat_id)
        
        markup = types.InlineKeyboardMarkup()
        for prod in prod_list:
            markup.add(types.InlineKeyboardButton(prod['name'], callback_data=f"prod_{prod.doc_id}"))
        markup.add(types.InlineKeyboardButton("⬅️ Назад до всіх категорій", callback_data="back_cat"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text=f"<b>📦 Категорія:</b> {cat['name']}\n<b>📃 Опис:</b> {cat['description']}",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в show_category_products: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при завантаженні товарів")


def show_product(call, prod_id):
    """Відображення інформації про товар"""
    try:
        prod = products.get(doc_id=prod_id)
        
        # Отримуємо опис без тексту товару (якщо він був доданий раніше)
        description = prod.get('description', '')
        # Видаляємо "Текст товару:" з опису, якщо він там є
        if "📄 Текст товару:" in description:
            description = description.split("📄 Текст товару:")[0].strip()
        
        stock = prod.get('stock', 0)
        stock_text = f"<b>📦 Наявність:</b> {stock} {'штука' if stock == 1 else 'штук' if stock < 5 else 'штук'}\n" if stock > 0 else ""
        
        markup = types.InlineKeyboardMarkup()
        markup.add(types.InlineKeyboardButton("🛒 Купити", callback_data=f"buy_{prod_id}"))
        markup.add(types.InlineKeyboardButton("⬅️ Назад", callback_data="back_prod"))
        markup.add(types.InlineKeyboardButton("⬅️ Назад до всіх категорій", callback_data="back_cat"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text=f"<b>🛍 Товар:</b> {prod['name']}\n<b>💰 Ціна:</b> {prod['price']} ₴\n<b>📃 Опис:</b> {description}\n{stock_text}",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в show_product: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при завантаженні товару")


def confirm_purchase(call, prod_id):
    """Підтвердження покупки"""
    try:
        prod = products.get(doc_id=prod_id)
        user = users.get(Query().user_id == call.from_user.id)
        
        if not user:
            bot.answer_callback_query(call.id, "❌ Користувача не знайдено")
            return
            
        price = float(prod['price'])
        balance = float(user.get('balance', 0.0))
        
        if balance < price:
            markup = types.InlineKeyboardMarkup()
            markup.add(types.InlineKeyboardButton("💰 Поповнити баланс", callback_data="topup"))
            
            bot.edit_message_text(chat_id=call.message.chat.id,
                                 message_id=call.message.message_id,
                                 text=f"💰 Це буде коштувати {price} ₴.\n"
                                      f"🚫 Недостатній баланс: {price - balance} ₴\n\n"
                                      f"⚡️ Поповніть баланс",
                                 reply_markup=markup)
            
            order_id = datetime.datetime.now().strftime('%Y%m%d%H%M%S')
            bot.send_message(call.message.chat.id,
                            f"❌ Замовлення: <code>#{order_id}</code> було скасовано",
                            parse_mode='HTML')
            bot.send_message(call.message.chat.id, "💣")
        else:
            markup = types.InlineKeyboardMarkup()
            markup.add(types.InlineKeyboardButton("✅ Так", callback_data="yes"))
            markup.add(types.InlineKeyboardButton("⬅️ Назад", callback_data=f"buy_{prod_id}"))
            
            bot.edit_message_text(chat_id=call.message.chat.id,
                                 message_id=call.message.message_id,
                                 text="<b>Ви впевнені, що хочете купити товар?</b>",
                                 parse_mode='HTML',
                                 reply_markup=markup)
            users.update({'pending_prod_id': prod_id}, Query().user_id == call.from_user.id)
    except Exception as e:
        logger.error(f"Помилка в confirm_purchase: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при обробці покупки")


def complete_purchase(call):
    """Завершення покупки"""
    try:
        user_id = call.from_user.id
        user = users.get(Query().user_id == user_id)
        
        if not user:
            bot.answer_callback_query(call.id, "❌ Користувача не знайдено")
            return
            
        prod_id = user.get('pending_prod_id')
        if not prod_id:
            bot.answer_callback_query(call.id, "❌ Товар не вибрано")
            return
            
        prod = products.get(doc_id=prod_id)
        price = float(prod['price'])
        balance = float(user.get('balance', 0.0))
        stock = prod.get('stock', 0)
        
        if balance < price:
            bot.answer_callback_query(call.id, "❌ Недостатній баланс")
            return
        
        if stock <= 0:
            bot.answer_callback_query(call.id, "❌ Товар закінчився")
            return
        
        # Відправка файлу або тексту товару (перед зменшенням stock)
        file_id = prod.get('file_id')
        file_to_send = None
        
        if isinstance(file_id, list):
            # Масив файлів - відправляємо перший доступний
            if file_id:
                file_to_send = file_id[0]
                # Видаляємо відправлений файл з масиву
                new_file_list = file_id[1:]  # Копія масиву без першого елемента
                # Оновлюємо масив файлів
                products.update({'file_id': new_file_list}, doc_ids=[prod_id])
        elif file_id and file_id.startswith('TEXT:'):
            # Текстовий товар - відправляємо текст покупцю
            file_to_send = file_id
        
        # Зменшуємо кількість товару
        products.update({'stock': stock - 1}, doc_ids=[prod_id])
        
        # Списування коштів та оновлення балансу
        new_balance = balance - price
        users.update({
            'balance': new_balance,
            'purchases': user.get('purchases', 0) + 1,
            'pending_prod_id': None
        }, Query().user_id == user_id)
        
        # Збереження замовлення
        orders.insert({
            'user_id': user_id,
            'product_name': prod['name'],
            'price': price,
            'date': datetime.datetime.now().strftime('%Y-%m-%d')
        })
        
        bot.send_message(call.message.chat.id, "🎉", message_effect_id=PARTY_EFFECT)
        
        # Відправка файлу або тексту товару
        if file_to_send:
            if isinstance(file_to_send, str) and file_to_send.startswith('TEXT:'):
                # Текстовий товар - відправляємо текст покупцю
                text_content = file_to_send[5:]  # Видаляємо префікс "TEXT:"
                bot.send_message(call.message.chat.id,
                               text_content,
                               parse_mode='HTML')
            else:
                # Файл - відправляємо файл
                send_file_by_id(call.message.chat.id, file_to_send)
        elif file_id:
            # Один файл (старий формат) - відправляємо файл
            send_file_by_id(call.message.chat.id, file_id)
        else:
            # Товар без файлу та тексту (не повинно бути, але на всяк випадок)
            bot.send_message(call.message.chat.id,
                           f"<b>📦 {prod['name']}</b>\n\n{prod.get('description', '')}",
                           parse_mode='HTML')
        
        bot.delete_message(call.message.chat.id, call.message.message_id)
        
        logger.info(f"Покупка завершена: user_id={user_id}, product={prod['name']}, price={price}")
    except Exception as e:
        logger.error(f"Помилка в complete_purchase: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при завершенні покупки")


def show_history(call):
    """Відображення історії замовлень"""
    try:
        user_id = call.from_user.id
        order_list = orders.search(Query().user_id == user_id)
        
        text = f"<b>Всього у вас замовлень: {len(order_list)}\n➖➖➖➖➖➖➖➖➖➖➖➖\n</b>"
        for order in order_list:
            text += f"{order['product_name']} – {order['price']} ₴ – {order['date']}\n➖➖➖➖➖➖➖➖➖➖➖➖\n"
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text=text,
                             parse_mode='HTML')
    except Exception as e:
        logger.error(f"Помилка в show_history: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при завантаженні історії")


def request_topup_amount(call):
    """Запит суми поповнення"""
    try:
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="<b>⚡️ Надішліть суму поповнення, мін. 1 ₴ (гривні):</b>",
                             parse_mode='HTML')
        bot.register_next_step_handler(call.message, process_amount)
    except Exception as e:
        logger.error(f"Помилка в request_topup_amount: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def process_amount(message):
    """Обробка введеної суми"""
    try:
        amount = float(message.text.replace(',', '.'))
        if amount < 1:
            bot.send_message(message.chat.id, "❌ Мінімальна сума 1 ₴")
            return
        
        bot.send_message(message.chat.id, "💰")
        
        markup = types.InlineKeyboardMarkup()
        markup.add(types.InlineKeyboardButton("💳 Оплата через AIFO", callback_data="aifo_pay"))
        
        bot.send_message(message.chat.id,
                        "<b>Виберіть спосіб поповнення:</b>",
                        parse_mode='HTML',
                        reply_markup=markup)
        
        users.update({'pending_amount': amount}, Query().user_id == message.from_user.id)
        logger.info(f"Запит на поповнення: user_id={message.from_user.id}, amount={amount}")
    except ValueError:
        bot.send_message(message.chat.id, "❌ Будь ласка, введіть число")
    except Exception as e:
        logger.error(f"Помилка в process_amount: {e}")
        bot.send_message(message.chat.id, "❌ Помилка при обробці суми")


def create_aifo_payment(call):
    """Створення платежу через AIFO API"""
    try:
        user_id = call.from_user.id
        user = users.get(Query().user_id == user_id)
        
        if not user:
            bot.answer_callback_query(call.id, "❌ Користувача не знайдено")
            return
        
        amount = float(user.get('pending_amount', 0))
        if amount < 1:
            bot.answer_callback_query(call.id, "❌ Некоректна сума")
            return
        
        # Перевірка конфігурації
        if not AIFO_SHOP_ID or not AIFO_SECRET_KEY or not AIFO_API_URL:
            error_msg = "❌ Помилка конфігурації: перевірте config.py"
            logger.error("Не налаштовано AIFO параметри в config.py")
            bot.edit_message_text(chat_id=call.message.chat.id,
                                 message_id=call.message.message_id,
                                 text=error_msg)
            return
        
        # Генерація унікального ID платежу
        invoice_id = int(time.time() * 1000) + user_id
        
        # Переконуємося що amount - це число (без зайвих знаків)
        amount = float(amount)
        
        # Створення підпису
        # create_signature сам форматує amount правильно (як PHP)
        sign = create_signature(AIFO_SHOP_ID, amount, AIFO_SECRET_KEY, invoice_id)
        
        # Створення платежу через AIFO API
        api_data = {
            'shop_id': AIFO_SHOP_ID,
            'amount': amount,
            'id': invoice_id,
            'sign': sign,
            'desc': f'Поповнення балансу @{user.get("username", f"user_{user_id}")}'
        }
        
        logger.info(f"Дані для API: {api_data}")
        
        logger.debug(f"Дані для API: shop_id={AIFO_SHOP_ID}, amount={amount}, id={invoice_id}, sign={sign[:32]}...")
        
        result = send_api_request('invoices/create', api_data)
        
        if not result:
            error_msg = "❌ Помилка при створенні платежу.\nПеревірте логи бота."
            bot.edit_message_text(chat_id=call.message.chat.id,
                                 message_id=call.message.message_id,
                                 text=error_msg)
            logger.error(f"Помилка створення платежу: user_id={user_id}, amount={amount}, result={result}")
            return
        
        # Перевірка наявності помилки
        if result.get('error'):
            error_text = result.get('error', 'Невідома помилка')
            error_data = result.get('error_data', '')
            error_msg = f"❌ Помилка API: {error_text}"
            if error_data:
                error_msg += f"\nДеталі: {error_data}"
            logger.error(f"API повернув помилку: {error_text}, дані: {error_data}")
            bot.edit_message_text(chat_id=call.message.chat.id,
                                 message_id=call.message.message_id,
                                 text=error_msg)
            return
        
        created_invoice_id = result.get('invoice_id')
        if not created_invoice_id:
            logger.error(f"API не повернув invoice_id: {result}")
            bot.edit_message_text(chat_id=call.message.chat.id,
                                 message_id=call.message.message_id,
                                 text="❌ Помилка: API не повернув ID платежу")
            return
        
        payment_url = result.get('payment_url', '')
        
        # Збереження invoice_id
        users.update({'invoice_id': created_invoice_id}, Query().user_id == user_id)
        
        markup = types.InlineKeyboardMarkup()
        if payment_url:
            markup.add(types.InlineKeyboardButton("💳 Перейти до оплати", url=payment_url))
        markup.add(types.InlineKeyboardButton("✅ Перевірити оплату", callback_data="check"))
        markup.add(types.InlineKeyboardButton("❌ Скасувати оплату", callback_data="cancel"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text=f"<b>💳 Платіж створено!</b>\n\n"
                                  f"<b>Сума:</b> {amount} ₴\n"
                                  f"<b>ID платежу:</b> <code>{created_invoice_id}</code>\n\n"
                                  f"Очікуємо оплату...",
                             parse_mode='HTML',
                             reply_markup=markup)
        
        logger.info(f"Платіж створено: user_id={user_id}, invoice_id={created_invoice_id}, amount={amount}")
    except ValueError as e:
        logger.error(f"Помилка валідації в create_aifo_payment: {e}")
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="❌ Помилка: некоректні дані")
    except Exception as e:
        logger.error(f"Помилка в create_aifo_payment: {e}", exc_info=True)
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text=f"❌ Помилка при створенні платежу.\nДеталі в логах.")


def check_payment(call):
    """Перевірка статусу оплати через AIFO API"""
    try:
        user_id = call.from_user.id
        user = users.get(Query().user_id == user_id)
        
        if not user:
            bot.answer_callback_query(call.id, "❌ Користувача не знайдено")
            return
        
        invoice_id = user.get('invoice_id')
        if not invoice_id:
            bot.answer_callback_query(call.id, "❌ Платіж не знайдено")
            return
        
        amount = float(user.get('pending_amount', 0))
        
        amount_formatted = f"{float(amount):.2f}"
        
        sign_string = f"{AIFO_SHOP_ID}:{amount_formatted}:{AIFO_SECRET_KEY}:{invoice_id}"
        sign = hashlib.sha256(sign_string.encode()).hexdigest()
        
        api_data = {
            'shop_id': AIFO_SHOP_ID,
            'id': invoice_id,
            'sign': sign
        }
        
        result = send_api_request('telegram/check', api_data, method='GET')
        
        if not result:
            bot.answer_callback_query(call.id, "❌ Помилка при перевірці платежу", show_alert=True)
            return
        
        if result.get('error'):
            error_text = result.get('error', 'Невідома помилка')
            logger.error(f"Помилка API при перевірці: {error_text}")
            bot.answer_callback_query(call.id, f"❌ Помилка: {error_text}", show_alert=True)
            return
        
        status = result.get('status', 0)
        status_text = result.get('status_text', 'unknown')
        
        if status == 1 and status_text == 'paid':
            invoice_amount = float(result.get('amount', amount))
            
            send_payment_webhook(user_id, invoice_id, invoice_amount)
            
            users.update({
                'pending_amount': 0,
                'invoice_id': None
            }, Query().user_id == user_id)
            
            bot.edit_message_text(chat_id=call.message.chat.id,
                                 message_id=call.message.message_id,
                                 text=f"<b>✅ Платіж оплачено!</b>\n\n"
                                      f"<b>Сума:</b> <code>{invoice_amount} ₴</code>\n\n"
                                      f"Кошти зараховано на баланс в системі AIFO.",
                                 parse_mode='HTML')
            
            logger.info(f"Платіж підтверджено: user_id={user_id}, invoice_id={invoice_id}, amount={invoice_amount}")
        else:
            bot.answer_callback_query(call.id, "⏳ Платіж ще не оплачено", show_alert=True)
    except Exception as e:
        logger.error(f"Помилка в check_payment: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при перевірці платежу")


def send_payment_webhook(user_id, invoice_id, amount):
    """Відправка webhook про оплату до AIFO"""
    try:
        user = users.get(Query().user_id == user_id)
        username = user.get('username', f"user_{user_id}")
        
        # Створення підпису для webhook
        sign = create_signature(AIFO_SHOP_ID, amount, AIFO_SECRET_KEY, invoice_id)
        
        webhook_data = {
            'shop_id': AIFO_SHOP_ID,
            'id': invoice_id,
            'amount': amount,
            'sign': sign,
            'telegram_user_id': user_id,
            'telegram_username': username
        }
        
        result = send_api_request('telegram/webhook', webhook_data)
        
        if result and not result.get('error'):
            # ✅ КРИТИЧНО: Оновлюємо баланс в database.json ПІСЛЯ успішного webhook
            # Отримуємо поточний баланс з database.json
            current_balance = float(user.get('balance', 0.0))
            new_balance = current_balance + float(amount)
            
            # Оновлюємо баланс в боті
            users.update({
                'balance': new_balance
            }, Query().user_id == user_id)
            
            logger.info(f"Webhook успішно відправлено: user_id={user_id}, invoice_id={invoice_id}")
            logger.info(f"Баланс оновлено в боті: {current_balance} ₴ → {new_balance} ₴")
            return True
        else:
            error_msg = result.get('error', 'Невідома помилка') if result else 'Немає відповіді'
            logger.error(f"Помилка відправки webhook: user_id={user_id}, invoice_id={invoice_id}, error={error_msg}")
            return False
    except Exception as e:
        logger.error(f"Помилка в send_payment_webhook: {e}")
        return False


def cancel_payment(call):
    """Скасування платежу"""
    try:
        user_id = call.from_user.id
        users.update({'pending_amount': 0, 'invoice_id': None}, Query().user_id == user_id)
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="❌ Платіж скасовано")
        logger.info(f"Платіж скасовано: user_id={user_id}")
    except Exception as e:
        logger.error(f"Помилка в cancel_payment: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def handle_admin(call):
    """Обробка адмін команд"""
    try:
        data = call.data
        
        if data == "admin_stats":
            show_stats(call)
        elif data.startswith("admin_products_"):
            page = int(data.split("_")[2])
            show_products_admin(call, page)
        elif data == "admin_add_cat":
            add_category_step1(call)
        elif data == "admin_del_cat":
            delete_category_step1(call)
        elif data == "admin_add_prod":
            add_product_step1(call)
        elif data == "admin_del_prod":
            delete_product_step1(call)
        elif data == "admin_back":
            admin_panel_inline(call)
    except Exception as e:
        logger.error(f"Помилка в handle_admin: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def admin_panel_inline(call):
    """Адмін панель (inline)"""
    try:
        markup = types.InlineKeyboardMarkup()
        markup.add(types.InlineKeyboardButton("📊 Статистика", callback_data="admin_stats"))
        markup.add(types.InlineKeyboardButton("📦 Товари", callback_data="admin_products_1"))
        markup.add(types.InlineKeyboardButton("➕ Додати товар", callback_data="admin_add_prod"),
                  types.InlineKeyboardButton("🗑 Видалити товар", callback_data="admin_del_prod"))
        markup.add(types.InlineKeyboardButton("➕ Додати категорію", callback_data="admin_add_cat"),
                  types.InlineKeyboardButton("🗑 Видалити категорію", callback_data="admin_del_cat"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="🔧 <b>Адмін меню</b>",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в admin_panel_inline: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def show_stats(call):
    """Відображення статистики"""
    try:
        today = datetime.datetime.now().strftime('%Y-%m-%d')
        week_ago = (datetime.datetime.now() - datetime.timedelta(days=7)).strftime('%Y-%m-%d')
        month_ago = (datetime.datetime.now() - datetime.timedelta(days=30)).strftime('%Y-%m-%d')
        
        daily_topup = sum(t.get('amount', 0) for t in transactions.search((Query().date >= today) & (Query().type == 'topup'))) or 0
        weekly_topup = sum(t.get('amount', 0) for t in transactions.search((Query().date >= week_ago) & (Query().type == 'topup'))) or 0
        monthly_topup = sum(t.get('amount', 0) for t in transactions.search((Query().date >= month_ago) & (Query().type == 'topup'))) or 0
        total_topup = sum(t.get('amount', 0) for t in transactions.search(Query().type == 'topup')) or 0
        
        daily_users = len(users.search(Query().reg_date >= today))
        weekly_users = len(users.search(Query().reg_date >= week_ago))
        monthly_users = len(users.search(Query().reg_date >= month_ago))
        total_users = len(users.all())
        
        markup = types.InlineKeyboardMarkup()
        markup.add(types.InlineKeyboardButton("⬅️ Назад", callback_data="admin_back"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text=f"<b>📊 Статистика\n\nПоповнення:</b>\n"
                                  f"<b>За добу:</b> <code>{daily_topup} ₴</code>\n"
                                  f"<b>За тиждень:</b> <code>{weekly_topup} ₴</code>\n"
                                  f"<b>За місяць:</b> <code>{monthly_topup} ₴</code>\n"
                                  f"<b>За весь час:</b> <code>{total_topup} ₴</code>\n\n"
                                  f"<b>Користувачі:</b>\n"
                                  f"<b>За добу:</b> <code>{daily_users}</code>\n"
                                  f"<b>За тиждень:</b> <code>{weekly_users}</code>\n"
                                  f"<b>За місяць:</b> <code>{monthly_users}</code>\n"
                                  f"<b>За весь час:</b> <code>{total_users}</code>",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в show_stats: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при завантаженні статистики")


def add_category_step1(call):
    """Крок 1: додавання категорії"""
    try:
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="<b>Введіть назву нової категорії</b>",
                             parse_mode='HTML')
        bot.register_next_step_handler(call.message, add_category_step2)
    except Exception as e:
        logger.error(f"Помилка в add_category_step1: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def add_category_step2(message):
    """Крок 2: додавання категорії"""
    try:
        name = message.text
        bot.send_message(message.chat.id, "<b>Введіть опис категорії</b>", parse_mode='HTML')
        bot.register_next_step_handler(message, add_category_step3, name)
    except Exception as e:
        logger.error(f"Помилка в add_category_step2: {e}")
        bot.send_message(message.chat.id, "❌ Помилка")


def add_category_step3(message, name):
    """Крок 3: додавання категорії"""
    try:
        description = message.text
        categories.insert({'name': name, 'description': description})
        bot.send_message(message.chat.id, "<b>✅ Категорію створено</b>", parse_mode='HTML')
        logger.info(f"Категорію додано: {name}")
    except Exception as e:
        logger.error(f"Помилка в add_category_step3: {e}")
        bot.send_message(message.chat.id, "❌ Помилка при створенні категорії")


def delete_category_step1(call):
    """Крок 1: видалення категорії"""
    try:
        cat_list = categories.all()
        
        markup = types.InlineKeyboardMarkup()
        for cat in cat_list:
            markup.add(types.InlineKeyboardButton(cat['name'], callback_data=f"delcat_{cat.doc_id}"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="<b>Виберіть категорію для видалення</b>",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в delete_category_step1: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def delete_category_step2(call):
    """Крок 2: видалення категорії"""
    try:
        cat_id = int(call.data.split("_")[1])
        markup = types.InlineKeyboardMarkup()
        markup.add(types.InlineKeyboardButton("✅ Так", callback_data=f"confdelcat_{cat_id}"))
        markup.add(types.InlineKeyboardButton("⬅️ Назад", callback_data="admin_del_cat"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="<b>Ви впевнені, що хочете видалити категорію?</b>",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в delete_category_step2: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def add_product_step1(call):
    """Крок 1: додавання товару"""
    try:
        cat_list = categories.all()
        
        markup = types.InlineKeyboardMarkup()
        for cat in cat_list:
            markup.add(types.InlineKeyboardButton(cat['name'], callback_data=f"addprodcat_{cat.doc_id}"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="<b>Виберіть категорію, де буде лежати товар</b>",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в add_product_step1: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def add_product_step2(call):
    """Крок 2: додавання товару"""
    try:
        cat_id = int(call.data.split("_")[1])
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="<b>Введіть назву товару</b>",
                             parse_mode='HTML')
        bot.register_next_step_handler(call.message, add_product_step3, cat_id)
    except Exception as e:
        logger.error(f"Помилка в add_product_step2: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def add_product_step3(message, cat_id):
    """Крок 3: додавання товару"""
    try:
        name = message.text
        bot.send_message(message.chat.id, "<b>Введіть опис товару</b>", parse_mode='HTML')
        bot.register_next_step_handler(message, add_product_step4, cat_id, name)
    except Exception as e:
        logger.error(f"Помилка в add_product_step3: {e}")
        bot.send_message(message.chat.id, "❌ Помилка")


def add_product_step4(message, cat_id, name):
    """Крок 4: додавання товару"""
    try:
        description = message.text
        bot.send_message(message.chat.id, "<b>Введіть скільки буде коштувати товар (₴)</b>", parse_mode='HTML')
        bot.register_next_step_handler(message, add_product_step5, cat_id, name, description)
    except Exception as e:
        logger.error(f"Помилка в add_product_step4: {e}")
        bot.send_message(message.chat.id, "❌ Помилка")


def add_product_step5(message, cat_id, name, description):
    """Крок 5: додавання товару"""
    try:
        price = float(message.text.replace(',', '.'))
        bot.send_message(message.chat.id,
                        "<b>Введіть кількість товарів для продажу</b>\n\n"
                        "Наприклад: 10 (буде доступно 10 товарів)",
                        parse_mode='HTML')
        bot.register_next_step_handler(message, add_product_step6, cat_id, name, description, price)
    except ValueError:
        bot.send_message(message.chat.id, "❌ Введіть число")
    except Exception as e:
        logger.error(f"Помилка в add_product_step5: {e}")
        bot.send_message(message.chat.id, "❌ Помилка")


def add_product_step6(message, cat_id, name, description, price):
    """Крок 6: додавання товару - введення кількості"""
    try:
        quantity = int(message.text)
        if quantity <= 0:
            bot.send_message(message.chat.id, "❌ Кількість має бути більше 0")
            return
        
        bot.send_message(message.chat.id,
                        f"<b>Кількість товарів: {quantity}</b>\n\n"
                        "<b>Надішліть файли або текст для товарів</b>\n\n"
                        "Підтримувані формати:\n"
                        "📄 Документи\n"
                        "🖼 Фото\n"
                        "🎥 Відео\n"
                        "🎵 Аудіо\n"
                        "🎤 Голосові повідомлення\n"
                        "📝 Або текст\n\n"
                        f"<i>Потрібно надіслати {quantity} {'файл' if quantity == 1 else 'файлів' if quantity < 5 else 'файлів'}</i>",
                        parse_mode='HTML')
        
        # Зберігаємо тимчасові дані для збору файлів
        user_id = message.from_user.id
        users.update({
            'adding_product': {
                'cat_id': cat_id,
                'name': name,
                'description': description,
                'price': price,
                'quantity': quantity,
                'files': []
            }
        }, Query().user_id == user_id)
        
        bot.register_next_step_handler(message, add_product_step7, quantity, 0)
    except ValueError:
        bot.send_message(message.chat.id, "❌ Введіть ціле число")
    except Exception as e:
        logger.error(f"Помилка в add_product_step6: {e}")
        bot.send_message(message.chat.id, "❌ Помилка")


def add_product_step7(message, total_quantity, current_count):
    """Крок 7: збір файлів для товару"""
    try:
        user_id = message.from_user.id
        user = users.get(Query().user_id == user_id)
        
        if not user or not user.get('adding_product'):
            bot.send_message(message.chat.id, "❌ Помилка: дані про товар не знайдено")
            return
        
        product_data = user['adding_product']
        file_id = None
        is_text = False
        
        # Обробка різних типів файлів
        if message.document:
            forwarded = bot.forward_message(CHANNEL_ID, message.chat.id, message.message_id)
            file_id = forwarded.document.file_id
        elif message.photo:
            forwarded = bot.forward_message(CHANNEL_ID, message.chat.id, message.message_id)
            file_id = forwarded.photo[-1].file_id
        elif message.video:
            forwarded = bot.forward_message(CHANNEL_ID, message.chat.id, message.message_id)
            file_id = forwarded.video.file_id
        elif message.audio:
            forwarded = bot.forward_message(CHANNEL_ID, message.chat.id, message.message_id)
            file_id = forwarded.audio.file_id
        elif message.voice:
            forwarded = bot.forward_message(CHANNEL_ID, message.chat.id, message.message_id)
            file_id = forwarded.voice.file_id
        elif message.video_note:
            forwarded = bot.forward_message(CHANNEL_ID, message.chat.id, message.message_id)
            file_id = forwarded.video_note.file_id
        elif message.text:
            # Текстовий товар
            file_id = "TEXT:" + message.text
            is_text = True
        
        if not file_id:
            bot.send_message(message.chat.id, "❌ Будь ласка, надішліть файл або текст")
            bot.register_next_step_handler(message, add_product_step7, total_quantity, current_count)
            return
        
        # Додаємо файл до списку
        product_data['files'].append(file_id)
        current_count += 1
        
        # Оновлюємо дані
        users.update({'adding_product': product_data}, Query().user_id == user_id)
        
        if current_count < total_quantity:
            # Потрібно ще файлів
            remaining = total_quantity - current_count
            bot.send_message(message.chat.id,
                           f"✅ Файл {current_count}/{total_quantity} отримано\n\n"
                           f"Залишилось надіслати: {remaining} {'файл' if remaining == 1 else 'файлів' if remaining < 5 else 'файлів'}",
                           parse_mode='HTML')
            bot.register_next_step_handler(message, add_product_step7, total_quantity, current_count)
        else:
            # Всі файли отримано - створюємо товар
            # Для текстових товарів використовуємо один текст
            if is_text and len(product_data['files']) == 1:
                # Текстовий товар - зберігаємо один товар з текстом
                products.insert({
                    'category_id': product_data['cat_id'],
                    'name': product_data['name'],
                    'description': product_data['description'],
                    'price': product_data['price'],
                    'file_id': product_data['files'][0],
                    'stock': 1  # Текстовий товар = 1 штука
                })
            else:
                # Файлові товари - зберігаємо всі файли в масиві
                products.insert({
                    'category_id': product_data['cat_id'],
                    'name': product_data['name'],
                    'description': product_data['description'],
                    'price': product_data['price'],
                    'file_id': product_data['files'],  # Масив файлів
                    'stock': len(product_data['files'])  # Кількість = кількість файлів
                })
            
            # Очищаємо тимчасові дані
            users.update({'adding_product': None}, Query().user_id == user_id)
            
            bot.send_message(message.chat.id, "✅")
            bot.send_message(message.chat.id,
                            f"<b>✅ Товар створено</b>\n\n"
                            f"Кількість: {len(product_data['files'])} {'штука' if len(product_data['files']) == 1 else 'штук'}",
                            parse_mode='HTML',
                            message_effect_id=PARTY_EFFECT)
            logger.info(f"Товар додано: {product_data['name']}, price={product_data['price']}, stock={len(product_data['files'])}")
    except Exception as e:
        logger.error(f"Помилка в add_product_step7: {e}", exc_info=True)
        users.update({'adding_product': None}, Query().user_id == user_id)
        bot.send_message(message.chat.id, f"❌ Помилка при створенні товару: {str(e)}")


def delete_product_step1(call):
    """Крок 1: видалення товару"""
    try:
        prod_list = products.all()
        
        markup = types.InlineKeyboardMarkup()
        for prod in prod_list:
            markup.add(types.InlineKeyboardButton(prod['name'], callback_data=f"delprod_{prod.doc_id}"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="<b>Виберіть товар, щоб його видалити</b>",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в delete_product_step1: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def delete_product_step2(call):
    """Крок 2: видалення товару"""
    try:
        prod_id = int(call.data.split("_")[1])
        markup = types.InlineKeyboardMarkup()
        markup.add(types.InlineKeyboardButton("✅ Так", callback_data=f"confdelprod_{prod_id}"))
        markup.add(types.InlineKeyboardButton("⬅️ Назад", callback_data="admin_del_prod"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text="<b>Ви впевнені, що хочете видалити товар?</b>",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в delete_product_step2: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка")


def show_products_admin(call, page):
    """Відображення товарів в адмін панелі"""
    try:
        prod_list = products.all()
        if not prod_list:
            bot.edit_message_text(chat_id=call.message.chat.id,
                                 message_id=call.message.message_id,
                                 text="<b>Товарів немає</b>",
                                 parse_mode='HTML')
            return
        
        total_pages = (len(prod_list) + 9) // 10
        page = max(1, min(page, total_pages))
        start = (page - 1) * 10
        end = min(start + 10, len(prod_list))
        
        prod = prod_list[start]
        cat = categories.get(doc_id=prod['category_id'])
        
        today = datetime.datetime.now().strftime('%Y-%m-%d')
        week_ago = (datetime.datetime.now() - datetime.timedelta(days=7)).strftime('%Y-%m-%d')
        month_ago = (datetime.datetime.now() - datetime.timedelta(days=30)).strftime('%Y-%m-%d')
        
        daily_sales = sum(o.get('price', 0) for o in orders.search((Query().product_name == prod['name']) & (Query().date >= today))) or 0
        weekly_sales = sum(o.get('price', 0) for o in orders.search((Query().product_name == prod['name']) & (Query().date >= week_ago))) or 0
        monthly_sales = sum(o.get('price', 0) for o in orders.search((Query().product_name == prod['name']) & (Query().date >= month_ago))) or 0
        total_sales = sum(o.get('price', 0) for o in orders.search(Query().product_name == prod['name'])) or 0
        
        markup = types.InlineKeyboardMarkup()
        if page > 1:
            markup.add(types.InlineKeyboardButton("⬅️", callback_data=f"admin_products_{page-1}"),
                      types.InlineKeyboardButton(f"{page}/{total_pages}", callback_data="noop"),
                      types.InlineKeyboardButton("➡️", callback_data=f"admin_products_{page+1}" if page < total_pages else "noop"))
        else:
            markup.add(types.InlineKeyboardButton("⬅️", callback_data="noop"),
                      types.InlineKeyboardButton(f"{page}/{total_pages}", callback_data="noop"),
                      types.InlineKeyboardButton("➡️", callback_data=f"admin_products_{page+1}" if page < total_pages else "noop"))
        markup.add(types.InlineKeyboardButton("⬅️ Назад", callback_data="admin_back"))
        
        bot.edit_message_text(chat_id=call.message.chat.id,
                             message_id=call.message.message_id,
                             text=f"<b>🛍 Товар:</b> {prod['name']}\n"
                                  f"<b>💰 Ціна:</b> {prod['price']} ₴\n"
                                  f"<b>📃 Опис:</b> {prod['description']}\n"
                                  f"<b>📦 Категорія:</b> {cat['name']}\n\n"
                                  f"<b>Статистика:</b>\n"
                                  f"<b>За добу:</b> <code>{daily_sales} ₴</code>\n"
                                  f"<b>За тиждень:</b> <code>{weekly_sales} ₴</code>\n"
                                  f"<b>За місяць:</b> <code>{monthly_sales} ₴</code>\n"
                                  f"<b>За весь час:</b> <code>{total_sales} ₴</code>",
                             parse_mode='HTML',
                             reply_markup=markup)
    except Exception as e:
        logger.error(f"Помилка в show_products_admin: {e}")
        bot.answer_callback_query(call.id, "❌ Помилка при завантаженні товарів")


if __name__ == "__main__":
    try:
        logger.info("Бот запущено")
        bot.polling(none_stop=True, interval=0, timeout=20)
    except Exception as e:
        logger.error(f"Критична помилка: {e}")
        raise
